<?php
/**
 * Файл отвечает за создание виртуального sitemap.xml.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Добавляет правило перезаписи для sitemap.xml.
 */
function sya_add_sitemap_rewrite_rule() {
    add_rewrite_rule( '^sitemap\.xml$', 'index.php?sitemap=true', 'top' );
}
add_action( 'init', 'sya_add_sitemap_rewrite_rule' );

/**
 * Регистрирует кастомную переменную запроса.
 */
function sya_add_sitemap_query_var( $vars ) {
    $vars[] = 'sitemap';
    return $vars;
}
add_action( 'query_vars', 'sya_add_sitemap_query_var' );

/**
 * Перехватывает запрос и генерирует sitemap.
 */
function sya_handle_sitemap_generation() {
    // Проверяем, был ли запрос именно к нашей sitemap.
    if ( get_query_var( 'sitemap' ) ) {
        // Получаем настройки: какие типы постов включать.
        $enabled_post_types = get_option('sya_sitemap_post_types', ['post' => '1', 'page' => '1']);
        
        // Получаем массив ключей (имен) типов постов.
        $post_types_to_include = !empty($enabled_post_types) ? array_keys($enabled_post_types) : [];
        
        // Если ничего не выбрано, выходим.
        if ( empty($post_types_to_include) ) {
            // Можно отдать пустой sitemap или 404
            status_header(404);
            die();
        }

        // --- Начинаем формирование XML ---
        header( 'Content-Type: text/xml; charset=utf-8' );
        echo '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
        echo '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";

        // 1. Добавляем главную страницу
        echo "\t<url>\n";
        echo "\t\t<loc>" . esc_url( home_url( '/' ) ) . "</loc>\n";
        echo "\t</url>\n";

        // 2. Делаем запрос к базе данных для получения всех нужных записей
        $query_args = [
            'post_type'      => $post_types_to_include,
            'post_status'    => 'publish',
            'posts_per_page' => -1, // Получить все посты
            'orderby'        => 'date',
            'order'          => 'DESC',
            // Оптимизация: не загружать лишние данные
            'no_found_rows'  => true,
            'update_post_meta_cache' => false,
            'update_post_term_cache' => false,
        ];
        
        $posts_query = new WP_Query( $query_args );

        if ( $posts_query->have_posts() ) {
            while ( $posts_query->have_posts() ) {
                $posts_query->the_post();
                
                $post_id = get_the_ID();
                $permalink = get_permalink( $post_id );
                $last_modified = get_the_modified_date( 'c', $post_id );

                echo "\t<url>\n";
                echo "\t\t<loc>" . esc_url( $permalink ) . "</loc>\n";
                echo "\t\t<lastmod>" . esc_html( $last_modified ) . "</lastmod>\n";
                echo "\t</url>\n";
            }
        }
        wp_reset_postdata(); // Сбрасываем запрос

        echo '</urlset>';
        die; // Завершаем выполнение
    }
}
add_action( 'template_redirect', 'sya_handle_sitemap_generation' );

/**
 * Очищает правила перезаписи при активации/деактивации плагина.
 */
// Получаем путь к главному файлу плагина динамически
$plugin_main_file = SYA_PLUGIN_PATH . 'seo-yandex-assistant.php';

function sya_flush_rewrite_rules_on_activate() {
    sya_add_sitemap_rewrite_rule();
    flush_rewrite_rules();
}
register_activation_hook( $plugin_main_file, 'sya_flush_rewrite_rules_on_activate' );

function sya_flush_rewrite_rules_on_deactivate() {
    flush_rewrite_rules();
}
register_deactivation_hook( $plugin_main_file, 'sya_flush_rewrite_rules_on_deactivate' );