<?php
/**
 * Файл отвечает за создание страницы настроек плагина
 * в меню админ-панели WordPress.
 */

// Прямой доступ к файлу запрещен
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Добавляет пункт меню для нашей страницы настроек.
 * Вызывается хуком 'admin_menu'.
 */
function sya_add_settings_page() {
    add_options_page(
        'SEO Yandex Assistant',          // Заголовок <title> страницы настроек
        'SEO Yandex Assistant',          // Название пункта в меню "Настройки"
        'manage_options',                // Права доступа, необходимые для просмотра страницы
        'sya-settings',                  // Уникальный slug (URL) страницы
        'sya_render_settings_page_html'  // Функция, которая выводит HTML-код страницы
    );
}
// Регистрируем нашу страницу в меню WordPress.
add_action( 'admin_menu', 'sya_add_settings_page' );


/**
 * Регистрирует настройки плагина с помощью Settings API.
 * Вызывается хуком 'admin_init'.
 */
function sya_register_plugin_settings() {
    // --- Секция API и Интеграций ---
    register_setting(
        'sya_options_group',            // Имя группы
        'sya_deepseek_api_key',         // Имя опции в базе данных (wp_options)
        'sya_sanitize_api_key'          // Функция для очистки значения перед сохранением
    );
    
    register_setting(
        'sya_options_group',
        'sya_yandex_metrika_id',
        'sya_sanitize_metrika_id'
    );

    add_settings_section(
        'sya_api_settings_section',     // Уникальный ID секции
        'Настройки API и Интеграций',   // Заголовок секции
        null,                           // Функция для вывода описания секции (не нужна)
        'sya-settings'                  // Slug страницы, на которой выводить секцию
    );

    add_settings_field(
        'sya_deepseek_api_key_field',          // Уникальный ID поля
        'DeepSeek AI API Key',                 // Название поля
        'sya_render_api_key_field_html',       // Функция для вывода HTML-кода поля
        'sya-settings',                        // Slug страницы
        'sya_api_settings_section'             // ID секции, к которой принадлежит поле
    );

    add_settings_field(
        'sya_yandex_metrika_id_field',
        'ID счетчика Яндекс.Метрики',
        'sya_render_metrika_id_field_html',
        'sya-settings',
        'sya_api_settings_section'
    );

    // --- Секция для robots.txt ---
    register_setting(
        'sya_options_group',
        'sya_robots_txt_content',
        'wp_kses_post' // Используем wp_kses_post для безопасного сохранения текста как есть
    );

    add_settings_section(
        'sya_robots_settings_section',
        'Настройки robots.txt',
        'sya_render_robots_section_description', // Функция для описания секции
        'sya-settings'
    );

    add_settings_field(
        'sya_robots_txt_content_field',
        'Содержимое robots.txt',
        'sya_render_robots_txt_field_html',
        'sya-settings',
        'sya_robots_settings_section'
    );
    
    // --- Секция для Sitemap ---
    add_settings_section(
        'sya_sitemap_settings_section',
        'Настройки XML-карты сайта (sitemap.xml)',
        'sya_render_sitemap_section_description',
        'sya-settings'
    );
    
    // Регистрируем опцию как массив. WordPress сам обработает сохранение.
    register_setting('sya_options_group', 'sya_sitemap_post_types');

    add_settings_field(
        'sya_sitemap_post_types_field',
        'Включить типы контента',
        'sya_render_sitemap_post_types_field_html',
        'sya-settings',
        'sya_sitemap_settings_section'
    );
}
// Регистрируем настройки при инициализации админ-панели.
add_action( 'admin_init', 'sya_register_plugin_settings' );


/**
 * HTML-код "обертки" страницы настроек.
 */
function sya_render_settings_page_html() {
    ?>
    <div class="wrap">
        <h1><?php echo esc_html( get_admin_page_title() ); ?></h1>
        <p>Здесь вы можете управлять глобальными настройками плагина SEO Yandex Assistant.</p>
        
        <form action="options.php" method="post">
            <?php
            settings_fields( 'sya_options_group' );
            do_settings_sections( 'sya-settings' );
            submit_button( 'Сохранить изменения' );
            ?>
        </form>
    </div>
    <?php
}

/**
 * Выводит HTML для поля API-ключа.
 */
function sya_render_api_key_field_html() {
    $api_key = get_option( 'sya_deepseek_api_key', '' );
    echo '<input type="password" name="sya_deepseek_api_key" value="' . esc_attr( $api_key ) . '" class="regular-text" placeholder="Введите ваш API ключ">';
    echo '<p class="description">Получите ваш ключ в личном кабинете <a href="https://platform.deepseek.com/" target="_blank">DeepSeek Platform</a>.</p>';
}

/**
 * Выводит HTML для поля ID Метрики.
 */
function sya_render_metrika_id_field_html() {
    $metrika_id = get_option( 'sya_yandex_metrika_id', '' );
    echo '<input type="text" name="sya_yandex_metrika_id" value="' . esc_attr( $metrika_id ) . '" class="regular-text" placeholder="Например, 90123456">';
    echo '<p class="description">Просто вставьте номер вашего счетчика.</p>';
}

/**
 * Выводит описание для секции robots.txt.
 */
function sya_render_robots_section_description() {
    echo '<p>Здесь вы можете редактировать виртуальный файл robots.txt. Он будет иметь приоритет над физическим файлом в корне сайта, если тот существует.</p>';
}

/**
 * Выводит HTML для поля редактирования robots.txt.
 */
function sya_render_robots_txt_field_html() {
    $content = get_option( 'sya_robots_txt_content' );
    if ( empty( $content ) ) {
        $sitemap_url = esc_url( home_url( '/sitemap.xml' ) );
        $host = isset($_SERVER['HTTP_HOST']) ? $_SERVER['HTTP_HOST'] : parse_url(home_url(), PHP_URL_HOST);
        
        $content = <<<ROBOTS
User-agent: *
Disallow: /wp-admin/
Disallow: /wp-includes/
Disallow: /cgi-bin/
Disallow: /?s=
Disallow: /search/
Disallow: */trackback/
Disallow: */feed/
Disallow: */comments/
Allow: /wp-admin/admin-ajax.php

User-agent: Yandex
Disallow: /wp-admin/
Disallow: /wp-includes/
Disallow: /cgi-bin/
Disallow: /?s=
Disallow: /search/
Disallow: */trackback/
Disallow: */feed/
Disallow: */comments/
Allow: /wp-admin/admin-ajax.php
Host: {$host}

Sitemap: {$sitemap_url}
ROBOTS;
    }

    echo '<textarea name="sya_robots_txt_content" class="large-text code" rows="15" style="font-family: monospace;">' . esc_textarea( $content ) . '</textarea>';
    echo '<p class="description">Не забудьте указать правильный путь к вашей Sitemap. По умолчанию это <code>/sitemap.xml</code>.</p>';
}

/**
 * Выводит описание для секции Sitemap.
 */
function sya_render_sitemap_section_description() {
    $sitemap_url = home_url( '/sitemap.xml' );
    echo '<p>Плагин автоматически создаст XML-карту сайта по адресу: <a href="' . esc_url($sitemap_url) . '" target="_blank">' . esc_url($sitemap_url) . '</a></p>';
    echo '<p>Выберите, какие типы контента должны быть включены в карту сайта.</p>';
}

/**
 * Выводит HTML для выбора типов записей в Sitemap.
 */
function sya_render_sitemap_post_types_field_html() {
    // Устанавливаем значения по умолчанию: Записи и Страницы включены.
    $options = get_option('sya_sitemap_post_types', ['post' => '1', 'page' => '1']); 
    
    // Получаем все публичные типы записей, зарегистрированные на сайте.
    $post_types = get_post_types(['public' => true], 'objects');
    
    foreach ($post_types as $post_type) {
        // Пропускаем 'attachment', так как страницы вложений почти никогда не нужны в sitemap.
        if ($post_type->name === 'attachment') {
            continue;
        }
        
        // Проверяем, отмечен ли чекбокс для данного типа записи.
        $checked = isset($options[$post_type->name]) ? 'checked' : '';
        
        echo '<fieldset>';
        echo '<label>';
        echo '<input type="checkbox" name="sya_sitemap_post_types[' . esc_attr($post_type->name) . ']" value="1" ' . $checked . '>';
        echo ' ' . esc_html($post_type->labels->name); // Выводим человеко-понятное название (например, "Товары" вместо "product")
        echo '</label>';
        echo '</fieldset>';
    }
}

/**
 * Функция очистки (санитизации) API-ключа перед сохранением.
 */
function sya_sanitize_api_key( $input ) {
    return sanitize_text_field( $input );
}

/**
 * Функция очистки ID Метрики.
 */
function sya_sanitize_metrika_id( $input ) {
    return preg_replace( '/[^0-9]/', '', $input );
}