<?php
/**
 * Файл отвечает за фоновую автоматическую SEO-оптимизацию через WP-Cron.
 */
if (!defined('ABSPATH')) exit;

// 1. Добавляем наш кастомный интервал для Cron
add_filter('cron_schedules', function($schedules) {
    if (!isset($schedules['sya_daily'])) {
        $schedules['sya_daily'] = ['interval' => DAY_IN_SECONDS, 'display' => 'Раз в сутки (SYA)'];
    }
    return $schedules;
});

// 2. Основная функция, которая будет выполняться по расписанию
function sya_process_seo_batch() {
    $options = get_option('sya_automation_settings');
    $api_key = get_option('sya_deepseek_api_key');

    // Проверяем, включена ли автоматизация, есть ли API-ключ и выбраны ли типы постов
    if (empty($options['enabled']) || empty($api_key) || empty($options['post_types'])) {
        return;
    }
    
    // Формируем запрос для поиска "необработанных" записей
    $query_args = [
        'post_type' => $options['post_types'],
        'post_status' => 'publish',
        'posts_per_page' => 5, // Обрабатываем по 5 штук за раз, чтобы не превысить лимиты и время выполнения
        'orderby' => 'modified',
        'order' => 'ASC',
    ];

    // Если не стоит галочка "Перезаписывать", ищем только посты, где наше SEO-поле НЕ существует
    if (empty($options['overwrite'])) {
        $query_args['meta_query'] = [
            [
                'key' => '_sya_seo_title',
                'compare' => 'NOT EXISTS'
            ]
        ];
    }
    
    $posts_to_process = new WP_Query($query_args);
    
    if ($posts_to_process->have_posts()) {
        while ($posts_to_process->have_posts()) {
            $posts_to_process->the_post();
            $post_id = get_the_ID();
            
            // Вызываем нашу центральную функцию для генерации и сохранения
            sya_generate_and_save_seo_data($post_id);
            
            // Небольшая пауза между запросами к API, чтобы не получить бан
            sleep(3); 
        }
    }
    wp_reset_postdata();
}
// Цепляем нашу функцию к кастомному хуку
add_action('sya_cron_hook', 'sya_process_seo_batch');


// 3. Управление расписанием при сохранении настроек
function sya_schedule_cron_on_settings_save($old_value, $value) {
    $is_enabled = !empty($value['enabled']);
    $is_scheduled = wp_next_scheduled('sya_cron_hook');

    if ($is_enabled && !$is_scheduled) {
        // Если опцию включили и задачи в расписании нет - добавляем ее
        wp_schedule_event(time() + 60, 'sya_daily', 'sya_cron_hook');
    } elseif (!$is_enabled && $is_scheduled) {
        // Если опцию выключили и задача в расписании есть - удаляем ее
        wp_clear_scheduled_hook('sya_cron_hook');
    }
}
// Цепляемся к хуку, который срабатывает при обновлении наших настроек
add_action('update_option_sya_automation_settings', 'sya_schedule_cron_on_settings_save', 10, 2);