<?php
/**
 * Вспомогательные функции для работы с AI.
 */

if ( ! defined( 'ABSPATH' ) ) exit;

/**
 * Генерирует SEO-предложения (title и description), но НЕ сохраняет их.
 *
 * @param int $post_id ID поста.
 * @return array|WP_Error Массив с ['seo_title', 'meta_description'] или объект WP_Error.
 */
function sya_get_ai_seo_suggestions( $post_id ) {
    $api_key = get_option( 'sya_deepseek_api_key' );
    if ( empty( $api_key ) ) {
        return new WP_Error( 'api_key_missing', 'API ключ DeepSeek не найден.' );
    }

    $post = get_post( $post_id );
    if ( ! $post ) {
        return new WP_Error( 'post_not_found', 'Пост с ID ' . $post_id . ' не найден.' );
    }
    
    $content = wp_strip_all_tags( strip_shortcodes( $post->post_content ) );
    if ( mb_strlen( $content ) < 50 ) {
        return new WP_Error( 'content_too_short', 'Контент поста слишком короткий для анализа.' );
    }
    
    $title = $post->post_title;
    $post_type = $post->post_type;

    $prompt_context = "Проанализируй следующий контент и сгенерируй для него SEO-данные на русском языке. Заголовок (seo_title) должен быть 50-70 символов, описание (meta_description) 150-200. Ответ верни СТРОГО в формате JSON: {\"seo_title\": \"...\", \"meta_description\": \"...\"}";

    if ( $post_type === 'product' && class_exists('WooCommerce') ) {
        $product = wc_get_product( $post_id );
        if ($product) {
            $price = $product->get_price();
            $attributes = strip_tags($product->get_attribute_summary());
            $prompt_context .= "\nЭто товар WooCommerce. Учти его название: '{$title}', цену: {$price}, характеристики: '{$attributes}'. Сделай акцент на коммерческой привлекательности, упомяни возможность покупки, доставки.";
        }
    } else {
        $prompt_context .= "\nЭто информационная страница/статья с названием: '{$title}'. Сделай акцент на кликабельности (CTR) и пользе для читателя.";
    }
    
    $prompt_context .= "\nКонтент для анализа: \n\"{$content}\"";
    
    $response = wp_remote_post('https://api.deepseek.com/chat/completions', [
        'timeout' => 60,
        'headers' => ['Content-Type' => 'application/json', 'Authorization' => 'Bearer ' . $api_key],
        'body' => json_encode(['model' => 'deepseek-chat', 'messages' => [['role' => 'user', 'content' => $prompt_context]], 'response_format' => ['type' => 'json_object']]),
    ]);

    if ( is_wp_error( $response ) ) {
        return $response;
    }

    $body = wp_remote_retrieve_body( $response );
    $data = json_decode( $body, true );
    $ai_json = $data['choices'][0]['message']['content'] ?? null;
    $ai_result = json_decode( $ai_json, true );

    if ( empty( $ai_result['seo_title'] ) || empty( $ai_result['meta_description'] ) ) {
        return new WP_Error( 'invalid_ai_response', 'AI вернул некорректные данные.', $ai_json );
    }

    return $ai_result;
}

/**
 * Использует предыдущую функцию для генерации и сразу сохраняет данные.
 * Используется для "однокликовой" оптимизации и Cron.
 *
 * @param int $post_id ID поста для обработки.
 * @return bool|WP_Error True в случае успеха, или объект WP_Error в случае ошибки.
 */
function sya_generate_and_save_seo_data( $post_id ) {
    $suggestions = sya_get_ai_seo_suggestions( $post_id );

    if ( is_wp_error( $suggestions ) ) {
        return $suggestions;
    }

    $seo_title = sanitize_text_field( $suggestions['seo_title'] );
    $meta_description = sanitize_textarea_field( $suggestions['meta_description'] );
    
    update_post_meta( $post_id, '_sya_seo_title', $seo_title );
    update_post_meta( $post_id, '_sya_meta_description', $meta_description );

    add_post_meta( $post_id, '_sya_og_title', $seo_title, true );
    add_post_meta( $post_id, '_sya_og_description', $meta_description, true );

    return true;
}