<?php
/**
 * Файл отвечает за регистрацию и отображение мета-бокса
 * на страницах редактирования записей.
 */

// Прямой доступ к файлу запрещен
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Регистрирует (добавляет) наш мета-бокс в WordPress.
 */
function sya_add_seo_meta_box() {
    $post_types = ['post', 'page', 'product'];

    add_meta_box(
        'sya_seo_metabox',
        'SEO-ассистент для Яндекса',
        'sya_render_meta_box_ui',
        $post_types,
        'advanced',
        'high'
    );
}

/**
 * Генерирует и выводит HTML-код содержимого мета-бокса.
 * @param WP_Post $post Объект текущего поста, который редактируется.
 */
function sya_render_meta_box_ui( $post ) {
    wp_nonce_field( 'sya_save_action', 'sya_nonce' );

    // Получаем все сохраненные мета-данные
    $seo_title           = get_post_meta( $post->ID, '_sya_seo_title', true );
    $meta_desc           = get_post_meta( $post->ID, '_sya_meta_description', true );
    $focus_keywords      = get_post_meta( $post->ID, '_sya_focus_keywords', true );
    $is_turbo_enabled    = get_post_meta( $post->ID, '_sya_is_turbo_enabled', true );
    $og_title            = get_post_meta( $post->ID, '_sya_og_title', true );
    $og_desc             = get_post_meta( $post->ID, '_sya_og_description', true );
    $og_image_id         = get_post_meta( $post->ID, '_sya_og_image_id', true );
    $og_image_url        = $og_image_id ? wp_get_attachment_image_url( $og_image_id, 'large' ) : '';
    $current_schema_type = get_post_meta( $post->ID, '_sya_schema_type', true );
    $faq_data            = get_post_meta( $post->ID, '_sya_schema_faq_data', true );
    ?>
    <div class="sya-container">
        <!-- Навигация по вкладкам -->
        <div class="sya-tabs">
            <a href="#sya-tab-main" class="sya-tab-link active">Основное</a>
            <a href="#sya-tab-analysis" class="sya-tab-link">Анализ контента</a>
            <?php if ( in_array( $post->post_type, ['page', 'product'] ) ) : ?>
            <a href="#sya-tab-commercial" class="sya-tab-link">Коммерческие факторы</a>
            <?php endif; ?>
            <a href="#sya-tab-schema" class="sya-tab-link">Микроразметка</a>
            <a href="#sya-tab-social" class="sya-tab-link">Соцсети (Open Graph)</a>
        </div>

        <!-- Контейнер для содержимого вкладок -->
        <div class="sya-tabs-content">

            <!-- Вкладка 1: Основное -->
            <div id="sya-tab-main" class="sya-tab-content active">
                <h4>Превью в Яндексе</h4>
                <div id="sya-snippet-preview" class="yandex-preview">
                     <div class="sya-preview-title"><?php echo esc_html( $seo_title ?: get_the_title( $post->ID ) ); ?></div>
                     <div class="sya-preview-url"><?php echo esc_url( get_permalink( $post->ID ) ); ?></div>
                     <div class="sya-preview-desc"><?php echo esc_html( $meta_desc ?: 'Здесь будет ваше мета-описание...' ); ?></div>
                </div>

                <div class="sya-field">
                    <label for="sya_seo_title"><strong>SEO-заголовок (Title)</strong></label>
                    <input type="text" id="sya_seo_title" name="sya_seo_title" value="<?php echo esc_attr( $seo_title ); ?>" class="widefat">
                    <div class="sya-progress-bar-container"><div id="sya-title-progress" class="sya-progress-bar"></div></div>
                    <p class="description">Рекомендуемая длина для Яндекса: 50-70 символов. Текущая: <span id="sya-title-counter">0</span></p>
                </div>

                <div class="sya-field">
                    <label for="sya_meta_description"><strong>Мета-описание (Description)</strong></label>
                    <textarea id="sya_meta_description" name="sya_meta_description" class="widefat" rows="4"><?php echo esc_textarea( $meta_desc ); ?></textarea>
                    <div class="sya-progress-bar-container"><div id="sya-desc-progress" class="sya-progress-bar"></div></div>
                    <p class="description">Рекомендуемая длина: 150-200 символов. Текущая: <span id="sya-desc-counter">0</span></p>
                </div>
                
                <div class="sya-field">
                    <label>
                        <input type="checkbox" name="sya_is_turbo_enabled" value="1" <?php checked( $is_turbo_enabled, '1' ); ?>>
                        Включить Турбо-страницу для этой записи
                    </label>
                    <p class="description">Требует дополнительной настройки на стороне сервера и в Яндекс.Вебмастере.</p>
                </div>
            </div>

            <!-- Вкладка 2: Анализ контента -->
            <div id="sya-tab-analysis" class="sya-tab-content">
                <div class="sya-field">
                    <label for="sya_focus_keywords"><strong>Фокусные ключевые слова (через запятую)</strong></label>
                    <input type="text" id="sya_focus_keywords" name="sya_focus_keywords" value="<?php echo esc_attr( $focus_keywords ); ?>" class="widefat">
                    <p class="description">Эти слова будут использоваться AI для генерации предложений и анализа текста.</p>
                </div>
                <div class="sya-ai-controls">
                    <button type="button" id="sya-generate-ai" class="button button-primary button-large">
                        <span class="dashicons dashicons-superhero" style="vertical-align: middle; margin-right: 5px;"></span>Сгенерировать SEO-данные
                    </button>
                    <button type="button" id="sya-analyze-content" class="button button-secondary" style="margin-left: 10px;">
                        <span class="dashicons dashicons-search" style="vertical-align: middle; margin-right: 5px;"></span>Провести углубленный анализ
                    </button>
                    <span id="sya-spinner" class="spinner"></span>
                </div>
                <div id="sya-ai-status" class="sya-ai-status-box">
                    <p>Нажмите "Анализировать", чтобы AI провел глубокий анализ текста, предложил LSI-ключи и дал конкретные рекомендации по улучшению.</p>
                </div>
            </div>

            <!-- Вкладка 3: Коммерческие факторы -->
            <?php if ( in_array( $post->post_type, ['page', 'product'] ) ) : ?>
            <div id="sya-tab-commercial" class="sya-tab-content">
                <h4>Чек-лист коммерческих факторов Яндекса</h4>
                <p class="description">Убедитесь, что на странице и на сайте в целом присутствуют эти элементы. Это критически важно для ранжирования коммерческих сайтов в Яндексе.</p>
                <ul class="sya-checklist">
                    <li><span class="dashicons dashicons-yes"></span> Наличие контактного телефона в шапке/подвале сайта.</li>
                    <li><span class="dashicons dashicons-yes"></span> Наличие адреса и схемы проезда (для оффлайн-бизнеса).</li>
                    <li><span class="dashicons dashicons-yes"></span> Подробная информация о способах доставки и оплаты.</li>
                    <li><span class="dashicons dashicons-yes"></span> Наличие цен на товары/услуги.</li>
                    <li><span class="dashicons dashicons-yes"></span> Наличие отзывов о товаре/компании.</li>
                    <li><span class="dashicons dashicons-yes"></span> Страница "О компании".</li>
                </ul>
                <p><em>Этот список носит информационный характер для самопроверки.</em></p>
            </div>
            <?php endif; ?>

            <!-- Вкладка 4: Микроразметка -->
            <div id="sya-tab-schema" class="sya-tab-content">
                <h4>Настройки Schema.org (JSON-LD)</h4>
                <p class="description">Выберите тип разметки для этой страницы. Плагин попытается автоматически заполнить данные из контента. Для "Вопрос-Ответ" появятся дополнительные поля.</p>
                
                <?php
                // Авто-определение типа, если он не сохранен
                if ( empty( $current_schema_type ) ) {
                    $post_type = get_post_type( $post->ID );
                    if ( $post_type === 'product' ) {
                        $current_schema_type = 'Product';
                    } elseif ( $post_type === 'post' ) {
                        $current_schema_type = 'Article';
                    } else {
                        $current_schema_type = 'WebPage';
                    }
                }
                ?>

                <div class="sya-field">
                    <label for="sya_schema_type"><strong>Тип разметки</strong></label>
                    <select id="sya_schema_type" name="sya_schema_type" class="widefat">
                        <option value="None" <?php selected($current_schema_type, 'None'); ?>>Отключить</option>
                        <option value="Article" <?php selected($current_schema_type, 'Article'); ?>>Статья (Article)</option>
                        <option value="WebPage" <?php selected($current_schema_type, 'WebPage'); ?>>Веб-страница (WebPage)</option>
                        <option value="Product" <?php selected($current_schema_type, 'Product'); ?>>Товар (Product)</option>
                        <option value="FAQPage" <?php selected($current_schema_type, 'FAQPage'); ?>>Вопрос-Ответ (FAQ)</option>
                    </select>
                </div>

                <!-- Динамический блок для FAQ -->
                <div id="sya-faq-fields" class="sya-schema-fields-container" style="<?php echo $current_schema_type === 'FAQPage' ? '' : 'display: none;'; ?>">
                    <div class="sya-field">
                        <label for="sya_schema_faq_data"><strong>Вопросы и ответы</strong></label>
                        <textarea id="sya_schema_faq_data" name="sya_schema_faq_data" class="widefat" rows="8" placeholder="Каждый новый вопрос с новой строки. Разделитель между вопросом и ответом - два символа ||"><?php echo esc_textarea( $faq_data ); ?></textarea>
                        <p class="description">
                            Формат: <strong>Вопрос? || Ответ на вопрос.</strong><br>
                            <em>Пример:</em><br>
                            <code>Сколько стоит доставка? || Доставка по Рыбинску бесплатная при заказе от 2000 рублей.</code><br>
                            <code>Есть ли у вас гарантия? || Да, на все товары предоставляется официальная гарантия производителя.</code>
                        </p>
                    </div>
                </div>
            </div>

            <!-- Вкладка 5: Соцсети (Open Graph) -->
            <div id="sya-tab-social" class="sya-tab-content">
                <h4>Превью при репосте в соцсетях</h4>
                <p class="description">Эти данные будут использоваться ВКонтакте, Telegram и другими сервисами при создании превью ссылки. Если поля пустые, будут использованы данные из вкладки "Основное" и миниатюра записи.</p>
                
                <div class="sya-field">
                    <label for="sya_og_title"><strong>Заголовок для соцсетей (og:title)</strong></label>
                    <input type="text" id="sya_og_title" name="sya_og_title" value="<?php echo esc_attr($og_title); ?>" class="widefat">
                </div>
                <div class="sya-field">
                    <label for="sya_og_description"><strong>Описание для соцсетей (og:description)</strong></label>
                    <textarea id="sya_og_description" name="sya_og_description" class="widefat" rows="3"><?php echo esc_textarea($og_desc); ?></textarea>
                </div>
                <div class="sya-field">
                    <label for="sya_og_image"><strong>Картинка для соцсетей (og:image)</strong></label>
                    <div class="sya-image-uploader">
                        <div class="sya-image-preview">
                            <?php if ($og_image_url): ?>
                                <img src="<?php echo esc_url($og_image_url); ?>" alt="Preview" style="max-width:300px; height:auto;">
                            <?php endif; ?>
                        </div>
                        <input type="hidden" id="sya_og_image_id" name="sya_og_image_id" value="<?php echo esc_attr($og_image_id); ?>">
                        <button type="button" class="button" id="sya_upload_og_image_button">Выбрать/Загрузить изображение</button>
                        <button type="button" class="button button-link-delete" id="sya_remove_og_image_button" style="<?php echo $og_image_id ? '' : 'display:none;'; ?>">Удалить изображение</button>
                        <p class="description">Рекомендуемый размер: 1200x630 пикселей.</p>
                    </div>
                </div>
            </div>

        </div>
    </div>
    <?php
}