<?php
/**
 * Файл отвечает за вывод всех SEO-тегов и скриптов на фронтенде сайта.
 * Использует метод перехвата буфера вывода для замены <title>.
 */

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// --- 1. ЗАПУСК БУФЕРИЗАЦИИ ---
function sya_start_output_buffering() {
    if ( is_singular() ) {
        ob_start( 'sya_buffer_callback_replace_title' );
    }
}
add_action( 'template_redirect', 'sya_start_output_buffering' );

// --- 2. ФУНКЦИЯ ОБРАБОТКИ БУФЕРА ---
function sya_buffer_callback_replace_title( $buffer ) {
    if ( ! is_singular() ) {
        return $buffer;
    }

    $post_id = get_queried_object_id();
    $seo_title = get_post_meta( $post_id, '_sya_seo_title', true );
    
    if ( ! empty( $seo_title ) ) {
        $seo_title = esc_html( $seo_title );
        $buffer = preg_replace( '/<title>(.*?)<\/title>/is', "<title>{$seo_title}</title>", $buffer, 1 );
    }
    
    return $buffer;
}

/**
 * Главная функция, которая добавляет остальные SEO-теги в <head> страницы.
 */
function sya_output_seo_tags() {
    if ( ! is_singular() ) {
        return;
    }
    
    $post_id = get_queried_object_id();

    // --- 3. Вывод мета-тега Description ---
    $meta_desc = get_post_meta( $post_id, '_sya_meta_description', true );
    if ( ! empty( $meta_desc ) ) {
        echo '<meta name="description" content="' . esc_attr( $meta_desc ) . '">' . "\n";
    }

    // --- 4. Подсказка для Турбо-страниц ---
    $is_turbo_enabled = get_post_meta( $post_id, '_sya_is_turbo_enabled', true );
    if ( $is_turbo_enabled ) {
        $turbo_content_url = get_permalink( $post_id );
        echo '<link rel="turbo:source" href="' . esc_url( $turbo_content_url ) . '">' . "\n";
    }

    // --- 5. Вывод Open Graph мета-тегов ---
    $og_title = get_post_meta($post_id, '_sya_og_title', true);
    $og_desc = get_post_meta($post_id, '_sya_og_description', true);
    $og_image_id = get_post_meta($post_id, '_sya_og_image_id', true);

    $final_og_title = !empty($og_title) ? $og_title : get_post_meta($post_id, '_sya_seo_title', true);
    if (empty($final_og_title)) {
        $final_og_title = get_the_title($post_id);
    }

    $final_og_desc = !empty($og_desc) ? $og_desc : get_post_meta($post_id, '_sya_meta_description', true);
    if (empty($final_og_desc)) {
        $final_og_desc = has_excerpt($post_id) ? get_the_excerpt($post_id) : wp_trim_words(get_the_content($post_id), 25);
    }

    $final_og_image = '';
    if ($og_image_id) {
        $final_og_image = wp_get_attachment_image_url($og_image_id, 'large');
    } elseif (has_post_thumbnail($post_id)) {
        $final_og_image = get_the_post_thumbnail_url($post_id, 'large');
    }

    echo "\n<!-- SEO Yandex Assistant Open Graph -->\n";
    echo '<meta property="og:type" content="article" />' . "\n";
    echo '<meta property="og:title" content="' . esc_attr($final_og_title) . '" />' . "\n";
    echo '<meta property="og:description" content="' . esc_attr(strip_tags($final_og_desc)) . '" />' . "\n";
    echo '<meta property="og:url" content="' . esc_url(get_permalink($post_id)) . '" />' . "\n";
    echo '<meta property="og:site_name" content="' . esc_attr(get_bloginfo('name')) . '" />' . "\n";
    if ($final_og_image) {
        echo '<meta property="og:image" content="' . esc_url($final_og_image) . '" />' . "\n";
    }
    echo "<!-- / SEO Yandex Assistant Open Graph -->\n";

    // --- 6. Вывод микроразметки Schema.org JSON-LD ---
    $schema_type = get_post_meta( $post_id, '_sya_schema_type', true );

    if ( empty( $schema_type ) ) {
        $post_type = get_post_type( $post_id );
        if ( $post_type === 'product' ) { $schema_type = 'Product'; } 
        elseif ( $post_type === 'post' ) { $schema_type = 'Article'; } 
        else { $schema_type = 'WebPage'; }
    }

    if ( $schema_type === 'None' ) {
        return;
    }

    $schema_data = ['@context' => 'https://schema.org'];

    switch ( $schema_type ) {
        case 'Article':
            $schema_data['@type'] = 'Article';
            $schema_data['headline'] = get_the_title( $post_id );
            $schema_data['mainEntityOfPage'] = ['@type' => 'WebPage', '@id' => get_permalink( $post_id )];
            if (has_post_thumbnail($post_id)) $schema_data['image'] = get_the_post_thumbnail_url( $post_id, 'full' );
            $schema_data['datePublished'] = get_the_date( 'c', $post_id );
            $schema_data['dateModified'] = get_the_modified_date( 'c', $post_id );
            $author_id = get_post_field( 'post_author', $post_id );
            $schema_data['author'] = ['@type' => 'Person', 'name' => get_the_author_meta( 'display_name', $author_id )];
            break;

        case 'Product':
            if ( ! class_exists('WooCommerce') ) break;
            $product = wc_get_product( $post_id );
            if ( ! $product ) break;

            $schema_data['@type'] = 'Product';
            $schema_data['name'] = $product->get_name();
            $schema_data['description'] = wp_strip_all_tags( $product->get_short_description() ?: $product->get_description() );
            if ($product->get_image_id()) $schema_data['image'] = wp_get_attachment_image_url( $product->get_image_id(), 'full' );
            if ($product->get_sku()) $schema_data['sku'] = $product->get_sku();
            
            $schema_data['offers'] = [
                '@type' => 'Offer',
                'priceCurrency' => get_woocommerce_currency(),
                'price' => $product->get_price(),
                'url' => get_permalink( $post_id ),
                'availability' => 'https://schema.org/' . ( $product->is_in_stock() ? 'InStock' : 'OutOfStock' ),
            ];
            
            if ($product->get_review_count() > 0) {
                $schema_data['aggregateRating'] = [
                    '@type' => 'AggregateRating',
                    'ratingValue' => $product->get_average_rating(),
                    'reviewCount' => $product->get_review_count(),
                ];
            }
            break;
            
        case 'FAQPage':
            $schema_data['@type'] = 'FAQPage';
            $faq_raw_data = get_post_meta( $post_id, '_sya_schema_faq_data', true );
            if ( ! empty( $faq_raw_data ) ) {
                $faq_pairs = explode( "\n", trim($faq_raw_data) );
                $mainEntity = [];
                foreach( $faq_pairs as $pair ) {
                    $parts = explode( '||', $pair, 2 );
                    if ( count( $parts ) === 2 ) {
                        $question = trim($parts[0]);
                        $answer = trim($parts[1]);
                        if( !empty($question) && !empty($answer) ) {
                            $mainEntity[] = [
                                '@type' => 'Question', 'name' => $question,
                                'acceptedAnswer' => ['@type' => 'Answer', 'text' => $answer]
                            ];
                        }
                    }
                }
                if (!empty($mainEntity)) {
                    $schema_data['mainEntity'] = $mainEntity;
                }
            }
            break;
            
        case 'WebPage':
        default:
            $schema_data['@type'] = 'WebPage';
            $schema_data['name'] = get_the_title( $post_id );
            $schema_data['url'] = get_permalink( $post_id );
            break;
    }

    if ( isset( $schema_data['@type'] ) ) {
        echo "\n<!-- SEO Yandex Assistant Schema -->\n";
        echo '<script type="application/ld+json">' . json_encode( $schema_data, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES | JSON_PRETTY_PRINT ) . '</script>' . "\n";
        echo "<!-- / SEO Yandex Assistant Schema -->\n";
    }
}

/**
 * Добавляет код счетчика Яндекс.Метрики в подвал сайта.
 */
function sya_output_yandex_metrika_script() {
    $metrika_id = get_option( 'sya_yandex_metrika_id' );
    if ( empty( $metrika_id ) ) { return; }
    $metrika_id = preg_replace( '/[^0-9]/', '', $metrika_id );
    echo <<<HTML
<!-- Yandex.Metrika counter -->
<script type="text/javascript" >
   (function(m,e,t,r,i,k,a){m[i]=m[i]||function(){(m[i].a=m[i].a||[]).push(arguments)};
   m[i].l=1*new Date();
   for (var j = 0; j < document.scripts.length; j++) {if (document.scripts[j].src === r) { return; }}
   k=e.createElement(t),a=e.getElementsByTagName(t)[0],k.async=1,k.src=r,a.parentNode.insertBefore(k,a)})
   (window, document, "script", "https://mc.yandex.ru/metrika/tag.js", "ym");
   ym({$metrika_id}, "init", { clickmap:true, trackLinks:true, accurateTrackBounce:true, webvisor:true });
</script>
<noscript><div><img src="https://mc.yandex.ru/watch/{$metrika_id}" style="position:absolute; left:-9999px;" alt="" /></div></noscript>
<!-- /Yandex.Metrika counter -->
HTML;
}
add_action( 'wp_footer', 'sya_output_yandex_metrika_script' );