<?php
/**
 * Файл-обработчик AJAX-запросов для взаимодействия с DeepSeek AI.
 */

// Прямой доступ к файлу запрещен
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

// --- 1. Обработчик для углубленного анализа (из мета-бокса) ---
add_action( 'wp_ajax_sya_get_ai_suggestions', 'sya_handle_ai_suggestions' );

function sya_handle_ai_suggestions() {
    // Проверки безопасности и прав
    check_ajax_referer( 'sya_ajax_nonce', 'nonce' );
    $post_id = isset( $_POST['post_id'] ) ? intval( $_POST['post_id'] ) : 0;
    if (!$post_id || !current_user_can('edit_post', $post_id)) {
        wp_send_json_error(['message' => 'Недостаточно прав или неверный ID поста.']);
    }
    
    // Получение данных для промпта
    $keywords = isset( $_POST['keywords'] ) ? sanitize_text_field( $_POST['keywords'] ) : '';
    if ( empty( $keywords ) ) {
        wp_send_json_error( ['message' => 'Ошибка: Фокусные ключевые слова отсутствуют.'] );
    }

    $api_key = get_option( 'sya_deepseek_api_key' );
    if ( empty( $api_key ) ) {
        wp_send_json_error( ['message' => 'Ошибка: API ключ DeepSeek не найден.'] );
    }

    $post = get_post( $post_id );
    $content = wp_strip_all_tags( strip_shortcodes( $post->post_content ) );
    $title = $post->post_title;

    if ( mb_strlen( $content ) < 100 ) {
        wp_send_json_error( ['message' => 'Контент слишком короткий для анализа (менее 100 символов).'] );
    }

    $prompt = "
        Ты — ведущий SEO-редактор, специализирующийся на поисковой системе Яндекс. Твоя задача — провести глубокий анализ текста статьи и дать конкретные рекомендации по его улучшению для повышения позиций в поиске.
        Ответ должен быть СТРОГО в формате JSON.

        Исходные данные:
        - Оригинальный заголовок: \"{$title}\"
        - Основные ключевые слова: \"{$keywords}\"
        - Текст статьи для анализа: \"{$content}\"

        Проанализируй текст и верни JSON-объект со следующей структурой:
        {
          \"content_score\": {
            \"clarity\": \"Оценка (от 0 до 100) ясности и понятности текста для читателя\",
            \"completeness\": \"Оценка (от 0 до 100) полноты раскрытия темы. Учти, все ли аспекты основного ключа затронуты.\",
            \"commercial_intent\": \"Оценка (от 0 до 100) коммерческой убедительности текста (если применимо). Есть ли призывы к действию, информация о ценах, доставке и т.д.?\"
          },
          \"lsi_suggestions\": {
            \"missing_keywords\": [
              \"ключевое LSI-слово 1, которое отсутствует, но должно быть\",
              \"связанная фраза 2\",
              \"синоним 3\"
            ],
            \"explanation\": \"Краткое объяснение, почему добавление этих слов и фраз улучшит семантическое ядро статьи.\"
          },
          \"actionable_recommendations\": [
            \"Первая конкретная рекомендация. Например: 'Добавьте в текст абзац о 'сравнении моделей', используя фразы 'что лучше выбрать' и 'отзывы покупателей'.'.\",
            \"Вторая конкретная рекомендация. Например: 'Включите информацию о 'гарантии и доставке', чтобы усилить коммерческие факторы.'.\"
          ]
        }
    ";

    $response = wp_remote_post( 'https://api.deepseek.com/chat/completions', [
        'timeout'   => 60,
        'headers'   => ['Content-Type' => 'application/json', 'Authorization' => 'Bearer ' . $api_key],
        'body'      => json_encode( ['model' => 'deepseek-chat', 'messages' => [['role' => 'user', 'content' => $prompt]], 'response_format' => ['type' => 'json_object']] ),
    ] );

    if ( is_wp_error( $response ) ) {
        wp_send_json_error( ['message' => 'Ошибка WordPress при запросе к API: ' . $response->get_error_message()] );
    }

    $body = wp_remote_retrieve_body( $response );
    $data = json_decode( $body, true );
    if ( isset( $data['error'] ) ) {
        wp_send_json_error( ['message' => 'Ошибка API DeepSeek: ' . $data['error']['message']] );
    }
    if ( empty( $data['choices'][0]['message']['content'] ) ) {
        wp_send_json_error( ['message' => 'Не удалось получить анализ от API. Ответ не содержит данных.'] );
    }
    
    $ai_content_json = $data['choices'][0]['message']['content'];
    $ai_result = json_decode( $ai_content_json, true );
    if ( json_last_error() !== JSON_ERROR_NONE ) {
        wp_send_json_error( ['message' => 'API вернул некорректный JSON.', 'raw_json' => $ai_content_json] );
    }
    
    wp_send_json_success( $ai_result );
}


// --- 2. Обработчик для кнопки "Сгенерировать SEO-данные" в мета-боксе ---
add_action('wp_ajax_sya_get_suggestions_for_metabox', 'sya_handle_get_suggestions_for_metabox');

function sya_handle_get_suggestions_for_metabox() {
    // Проверки безопасности
    check_ajax_referer('sya_ajax_nonce', 'nonce');
    $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
    if (!$post_id || !current_user_can('edit_post', $post_id)) {
        wp_send_json_error(['message' => 'Недостаточно прав или неверный ID поста.']);
    }

    // Вызываем helper-функцию, которая только генерирует данные, но НЕ сохраняет их
    $result = sya_get_ai_seo_suggestions( $post_id );

    if ( is_wp_error( $result ) ) {
        // Если helper-функция вернула ошибку, передаем ее в AJAX ответ
        wp_send_json_error(['message' => $result->get_error_message()]);
    } else {
        // Если все успешно, отправляем сгенерированные title и description обратно в JavaScript
        wp_send_json_success( $result );
    }
}


// --- 3. Обработчик для "однокликовой" генерации из списка записей ---
add_action('wp_ajax_sya_generate_single_seo', 'sya_handle_generate_single_seo');

function sya_handle_generate_single_seo() {
    // Получаем ID поста из AJAX-запроса
    $post_id = isset($_POST['post_id']) ? intval($_POST['post_id']) : 0;
    
    // Проверяем nonce (ключ безопасности)
    check_ajax_referer('sya_generate_seo_' . $post_id, 'nonce');

    // Проверяем права пользователя и наличие ID поста
    if (!$post_id || !current_user_can('edit_post', $post_id)) {
        wp_send_json_error(['message' => 'Недостаточно прав или неверный ID поста.']);
    }

    // Вызываем helper-функцию, которая генерирует И СОХРАНЯЕТ данные
    $result = sya_generate_and_save_seo_data( $post_id );

    if ( is_wp_error( $result ) ) {
        // Если helper-функция вернула ошибку, передаем ее в AJAX ответ
        wp_send_json_error(['message' => $result->get_error_message()]);
    } else {
        // Если все прошло успешно, отправляем ответ об успехе
        wp_send_json_success(['message' => 'SEO-данные успешно сгенерированы и сохранены.']);
    }
}